//
//  MAStore.h
//  MAStore
//
//  Created by Stefan Fuerst on 22.04.18.
//  Copyright (c) 2018 Media Atelier. All rights reserved.
//

#import <CoreData/CoreData.h>
#import "StorageKitCache.h"
#import "StorageKitConstants.h"
#import "SeaCrypto.h"
#import "NSURL+StorageKit.h"


@class StorageKitCacheItem;
@class StorageKitChange;

@interface StorageKitStore : NSIncrementalStore

+(void)register;
+(BOOL)isCorrectFormat:(NSString *)path;
+(BOOL)canOpenURL:(NSURL *)URL withPassword:(NSString*)password;
+(NSString *)documentIdentifierForURL:(NSURL *)URL;
+(NSString *)fileExtension;
+(void)prepareNewStoreAtURL:(NSURL *)URL;
+(void)listAllAvailableFiles:(id)sender;
+(void)registerURL:(NSURL *)url withFileInfo:(NSDictionary *)info volumeInfo:(NSDictionary *)volumeInfo;

@property (nonatomic,strong) NSString * errorLevel;
@property (nonatomic,strong) NSDateFormatter * dateFormatter;
@property (nonatomic,strong) NSMutableDictionary * entityKeys;
@property (nonatomic,strong) NSMutableDictionary * reverseEntityKeys;
@property (nonatomic,strong) NSMutableDictionary * infoDictionary;
@property (nonatomic,strong) NSMutableDictionary * managedObjectIDs;
@property (nonatomic) BOOL recover;

@property (nonatomic,strong) NSDictionary * upToChange;

@property (nonatomic) BOOL hasRemoteFiles;

@property (nonatomic,strong) NSDictionary * entities;
@property (nonatomic,strong) NSDictionary * userInfo; /// Custom info in changes
@property (nonatomic,strong) NSString * session;
@property (nonatomic,strong) NSString * userName;
@property (nonatomic,strong) NSString * fullUserName;
@property (nonatomic,strong) NSString * osVersion;
@property (nonatomic,strong) NSString * machineName;
@property (nonatomic,strong) NSMutableSet * contexts;
@property (nonatomic,strong) NSURL * entitiesURL;
@property (nonatomic,strong) NSURL * changesURL;
@property (nonatomic,strong) NSURL * cacheURL;
@property (nonatomic,strong) NSURL * errorsURL;
@property (nonatomic,strong) StorageKitCache * cache;
@property (nonatomic,assign) FSEventStreamRef eventStream;
@property (nonatomic,strong) SeaCrypto * crypto;
@property (nonatomic,strong) NSMutableArray * log;
@property (nonatomic,strong) NSFetchRequest * lastRequest;
@property (nonatomic,strong) NSArray * lastResults;
@property (nonatomic,strong) NSDictionary * progressInfo;

@property (nonatomic,strong) NSOperationQueue * saveOperationQueue;

@property (nonatomic,strong) dispatch_queue_t readQueue;

@property (nonatomic,assign) BOOL closed;
@property (nonatomic,assign) BOOL updating;
@property (nonatomic,assign) BOOL blockSaves;

@property (nonatomic,assign) NSInteger saveOperationsCount;
@property (nonatomic,assign) NSInteger saveOperationsValue;

-(void)resetCache;
-(void)checkAllFiles;
-(void)preventFromSaving;
-(void)resetAttributesForManagedObject:(NSManagedObject *)managedObject;
-(StorageKitCacheItem *)loadValuesForObjectID:(NSManagedObjectID *)objectID withContext:(NSManagedObjectContext *)context;
-(StorageKitCacheItem *)loadValuesForObjectID:(NSManagedObjectID *)objectID withContext:(NSManagedObjectContext *)context upToChange:(NSString *)changeName;
-(id)executeFetchRequest:(NSFetchRequest *)request withContext:(NSManagedObjectContext *)context;
-(NSArray *)changesFilesFor:(NSEntityDescription *)entity withUID:(NSString *)uid;
-(NSString *)password;
-(NSString *)documentIdentifier;
-(id)encrypt:(id)value;
-(id)decrypt:(id)value;
-(NSString *)fileNameForEntity:(NSEntityDescription *)entity;
-(NSString *)entityNameForFileName:(NSString *)fileName;
-(void)writeAllObjectsInContext:(NSManagedObjectContext *)context fromStore:(StorageKitStore *)fromStore  completionBlock:(void(^)(NSManagedObjectContext * context))completion;
-(void)writeAllObjectsInContext:(NSManagedObjectContext *)context fromStore:(StorageKitStore *)fromStore options:(NSDictionary *)options completionBlock:(void(^)(NSManagedObjectContext * context))completion;
-(void)unobserveDirectory;
-(NSString *)uidForObjectID:(NSManagedObjectID *)objectID inContext:(NSManagedObjectContext *)context;
-(void)updateChanges;
-(NSDictionary *)entities;
-(NSString *)checkSum;
-(NSArray *)sessionInfo;
-(void)log:(NSString *)string;
-(void)log:(NSString *)string type:(NSString *)type;
-(NSArray *)infoArrayForContext:(NSManagedObjectContext *)context;
-(void)refreshInsertedObjects:(NSDictionary *)insertedRecords updatedObjects:(NSDictionary *)updatedRecords deletedObjects:(NSDictionary *)deletedRecords;
-(NSDictionary *)changesDictionaryForObjectIDs:(NSSet *)objectIDs;
-(void)setDocumentIdentifier:(NSString *)identifier;
-(NSDictionary *)validatedChanges:(NSDictionary *)aChange;
-(NSDictionary *)validatedChanges:(NSDictionary *)aChange lightWeight:(BOOL)lightweight;
-(NSMutableDictionary * )combineChangeDictionary:(NSMutableDictionary *)dict with:(NSMutableDictionary *)other;
-(void)logError:(NSError *)error domain:(NSString *)domain;
-(NSMutableDictionary *)sessionDictionary;
-(NSArray *)errorsArray;
-(NSURL *)correctedDataURL:(NSURL *)input withFileManager:(NSFileManager *)manager;
-(NSURL *)correctedDataURL:(NSURL *)input;
-(void)updateChangesWithCompletion:(void(^)(BOOL success, NSError * error))completionHandler;

-(NSString *)changesInfo;

@end


@interface MAIncrementalStoreNode : NSIncrementalStoreNode
//@property (nonatomic,strong) NSManagedObjectContext * context;
@property (nonatomic,assign) NSInteger valueCount;

@end


@interface NSManagedObjectContext (StorageKit)
-(NSManagedObject *)storeObjectWithURI:(id)uri;
-(NSArray *)storeObjectsWithURIs:(NSArray *)uris;
-(void)setIsMergingExternally:(BOOL)isMerging;
-(BOOL)isMergingExternally;
-(void)storageKitRefreshObject:(NSManagedObject *)object withStore:(StorageKitStore *)store;
@end



@interface NSObject (StorageKit)
-(NSArray *)arrayForStore;
@end

@interface NSSet (StorageKit)
-(NSArray *)arrayForStore;
@end

@interface NSOrderedSet (StorageKit)
-(NSArray *)arrayForStore;
@end

@interface NSManagedObject (StorageKit)
-(NSString *)storeUID;
-(void)saveToStore:(StorageKitStore *)store type:(NSString *)type;
-(void)saveToStore:(StorageKitStore *)store withChange:(StorageKitChange *)change type:(NSString *)type;
-(void)saveFromStore:(StorageKitStore *)fromStore toStore:(StorageKitStore *)toStore type:(NSString *)type;
-(void)saveFromStore:(StorageKitStore *)fromStore toStore:(StorageKitStore *)toStore withChange:(StorageKitChange *)change type:(NSString *)type;
-(void)saveFromStore:(StorageKitStore *)fromStore toStore:(StorageKitStore *)toStore withChange:(StorageKitChange *)change options:(NSDictionary *)options type:(NSString *)type;
-(id)storageKitCacheValueForKey:(NSString *)key;
-(void)setStorageKitCacheValue:(id)value forKey:(NSString *)key;
-(void)storageKitResetCacheValues;
-(void)storageKitDidRefresh;
-(void)storageKitCachePrepare;

@end


@interface NSManagedObjectID (StorageKit)

-(NSString *)storeSavePath;
-(NSString *)storeUID;
@end


@interface NSEntityDescription (StorageKit)

-(NSArray *)allSubentities;

@end



@interface StorageKitRequestItem : NSObject

@property (nonatomic,strong) NSManagedObject * managedObject;
@property (nonatomic,strong) NSManagedObjectID * objectID;
@property (nonatomic,strong) StorageKitCacheItem * cacheItem;
@property (nonatomic,strong) NSManagedObjectContext * context;
@property (nonatomic,strong) StorageKitStore * store;
-(NSDictionary *)dictionary;


@end

